/************************************************************************************************************\

Module Name:    DecoderModule.h

Description:    Module for Liberatus H264 video encoder.

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

#ifndef INC_DecoderModule_H
#define INC_DecoderModule_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "Liberatus.h"
#include "CommonUtils.h"
#include "ModuleThread.h"
#include "ModuleLink.h"
#include "LH264VideoCodec.h"

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Structure:      DecoderModule

Description:    Module for Liberatus H264 video decoder.

\************************************************************************************************************/
typedef struct tagDecoderModule
{
    LDeviceThread_Handle    hDevThreadIn;       // Input device thread
    LDeviceThread_Handle    hDevThreadOut;      // Output device thread
    LH264D_Handle           hDecoder;           // Encoder handle
    ModuleThread            oCpuThreadIn;       // Input CPU thread
    ModuleThread            oCpuThreadOut;      // Output CPU thread
    ModuleLinkInput         oInLink;            // Input module link
    ModuleLink              oOutLink;           // Output module link
    MBOOL                   bDecoding;          // Indicate if the decoder is running.
    MBOOL                   bNalBlobFlushed;    // Indicate that the last NAL blob has been flushed.
    MUINT64                 uiAccessUnitId;     // NAL blop unit id.
    MUINT64                 uiTotalSizeBytes;   // Total size in bytes decoded.
    MUINT64                 uiElapsedTimeUsec;  // Elapse usec since the first completed picture.
    MUINT                   uiPictureCount;     // Number of completed NALUs.
    MUINT                   uiFrameRateNum;     // Frame rate numerator.
    MUINT                   uiFrameRateDen;     // Frame rate denominator.
    MBOOL                   bDecodeNalBlobs;    // The decoding will be done with NAL blobs.
    MCHAR8                  szModuleName[16];

} DecoderModule;

#define DecoderModule_Construct { /*.hDevThreadIn       =*/ MNULL, \
                                  /*.hDevThreadOut      =*/ MNULL, \
                                  /*.hDecoder           =*/ MNULL, \
                                  /*.oCpuThreadIn       =*/ ModuleThread_Construct, \
                                  /*.oCpuThreadOut      =*/ ModuleThread_Construct, \
                                  /*.oInLink            =*/ ModuleLinkInput_Construct, \
                                  /*.oOutLink           =*/ ModuleLink_Construct, \
                                  /*.bDecoding          =*/ MFALSE, \
                                  /*.bNalBlobFlushed    =*/ MFALSE, \
                                  /*.uiAccessUnitId     =*/ 0, \
                                  /*/uiTotalSizeBytes   =*/ 0, \
                                  /*.uiElapsedTimeUsec  =*/ 0, \
                                  /*.uiPictureCount     =*/ 0, \
                                  /*.uiFrameRateNum     =*/ 0, \
                                  /*.uiFrameRateDen     =*/ 0, \
                                  /*.bDecodeNalBlobs    =*/ MFALSE, \
                                  /*.szModuleName       =*/ ""}

// -----------------------------------------------------------------------------------------------------------
//                  G L O B A L   V A R I A B L E / F U N C T I O N   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------
LStatus DecMod_Init(DecoderModule*  poDecMod,
                    LDevice_Handle  hDevice,
                    MUINT           uiOutBufferCount,
                    MUINT           uiFrameRateNum,
                    MUINT           uiFrameRateDen,
                    MBOOL           bDecodeNalBlobs,
                    MBOOL           bReduceLocalMemUsage);
void DecMod_Cleanup(DecoderModule* poDecMod);
LStatus DecMod_Start(DecoderModule* poDecMod);
void DecMod_Stop(DecoderModule* poDecMod);

// -----------------------------------------------------------------------------------------------------------
//                                 I N L I N E S   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------

#endif // INC_DecoderModule_H
